<?	// copyright (C) 2005 - Matt Newberry, matt@mattnewberry.net
	// all rights reserved, except as licensed under contract
	
	/*  A Calendar is a data structure holding an array of Event objects; each Event contains an expiration 
		date code, a display date (which may be any suitable text) and a description. Events are ordered ascending 
		by date code.
	
		Public Interface:
		
				//-- constructor ---------------
			
			function Calendar($filename)
				- creates a Calendar from an XML datafile; $filename should be a full path 
					relative to the executing script directory
			
				//-- display functions ---------------
			
			function getNextEvent();
				- returns the next Event sequentially; returns null if no more Events
			
			function Event.printEventID();
				- prints the Event's identifier; mainly for use as a url parameter
			
			function Event.printDate();
				- prints the Event's display date, html-encoded
			
			function Event.printDescription();
				- prints the Event's description, html-encoded
			
			function Event.printDescriptionTruncated($maxlen);
				- prints the Event's description truncated to $maxlen chars, html-encoded
			
				//-- error handling ---------------
			
			function parseFailed();
				- returns false if the Calendar was successfully parsed from xml, otherwise true (from XmlSimpleBaseClass)
			
			function getParseError();
				- returns error code from the PHP xml parser; 0 if no error, 8001 if input file not found 
					or not readable (from XmlSimpleBaseClass)
			
			function getParseMessage();
				- returns displayable error message from the PHP xml parser (from XmlSimpleBaseClass)
			
				//-- admin functions ---------------
			
			function getEvent($key);
				- returns an Event by it's internal identifier; may return null
			
			function addEvent($expires, $date, $description);
				- add an Event in order by expiration date
			
			function deleteEvent($key);
				- delete the Event with the given key
			
			function Event.setExpiration($exp);
				- set the Event's expiration date, which must be a legal date
			
			function Event.setDisplayDate($date);
				- set the Event's display date, which can be any meaningful string
			
			function Event.setDescription($desc);
				- set the Event's description
			
			function save();
				- save data back to disk under original filename (from XmlSimpleBaseClass)
		
		
		Typical usage in HTML display context:
		
			$calendar = new Calendar("data/calendar.xml");
			if (! $calendar->parseFailed()) {
				while (($e = $calendar->getNextEvent()) != null) {
					$e->printDate();
					$e->printDescription();
				}
			}
		
		
		XML Data Format:
		
			<calendar>
				<events>
					<event>
						<expires>INTEGER-DATE-CODE</expires>
						<displaydate>TEXT</displaydate>
						<description>TEXT</description>
					</event>
						.
						.
				</events>
			</calendar>
	*/
	
	require_once("XmlSimpleBaseClass.class.php");
	
	class Calendar extends XmlSimpleBaseClass {
		var $_event;
		var $pm_events = array();
		var $_count = 0;
		var $_months = array("January","February","March","April","May","June",
								"July","August","September","October","November","December");
		
		function Calendar($filename) {
			$this->XmlSimpleBaseClass($filename);
			$this->_count = count($this->pm_events);
		}
		
		 function _p_start_element($parser, $element, &$attributes) {
		 	parent::_p_start_element($parser, $element, &$attributes);
		 	if ($element == "event") {
				$this->_event = new Event();
			}
		 }
		
		function _p_cdata($parser, $text) {
			switch ($this->_tag) {
				case "expires":
					$this->_event->setExpiration($text);
				break;
				case "displaydate":
					$this->_event->setDisplayDate($text);
					break;
				case "description":
					$this->_event->setDescription($text);
					break;
			}
		}
		
		 function _p_end_element($parser, $element) {
		 	parent::_p_end_element($parser, $element);
		 	if ($element == "event") {
				if ($this->_event->pm_expires > time())		// filter expired events
					$this->pm_events[$this->_event->pm_expires] = $this->_event;
			}
		 	else if ($element == "events") {
				ksort($this->pm_events);
			}
		 }
		
		function getNextEvent() {
			if (list($key, $event) = each($this->pm_events)) {
				return $event;
			}
			else {
				reset($this->pm_events);
				return null;
			}
		}
		
		function getEvent($key) {
			return $this->pm_events[$key];
		}
		
		function addEvent($expires, $date, $description) {
			$event = new Event();
			while (array_key_exists($expires, $this->pm_events))
				++$expires;
			$event->setExpiration($expires);
			$event->setDisplayDate($date);
			$event->setDescription($description);
			$this->pm_events[$expires] = $event;
			ksort($this->pm_events);
		}
		
		function deleteEvent($key) {
			foreach ($this->pm_events as $e) {
				if ($e->pm_expires == $key)
					unset($this->pm_events[$key]);
			}
			ksort($this->pm_events);
		}
		
		function monthName($monthnum) {
			return $this->_months[$monthnum-1];
		}
	}
	
	class Event {
		var $pm_expires;
		var $pm_displaydate;
		var $pm_description;
		
		function Event() {
			;
		}
		
		function setExpiration($exp) {
			$this->pm_expires = $exp;
		}
		
		function setDisplayDate($date) {
			$this->pm_displaydate .= $date;
		}
		
		function setDescription($desc) {
			$this->pm_description .= $desc;
		}
		
		function printEventID() {
			print htmlspecialchars($this->pm_expires);
		}
		
		function printDate() {
			print htmlspecialchars($this->pm_displaydate);
			flush();
		}
		
		function printDescription() {
			print htmlspecialchars($this->pm_description);
			flush();
		}
		
		function printDescriptionTruncated($maxlen) {
			print htmlspecialchars(substr($this->pm_description, 0, $maxlen));
			if (strlen($this->pm_description) > $maxlen)
				print "...";
		}
	}
?>
